///////////////////////////////////////////////////////////////////////////////////
//No part of this file can be copied or released without the consent of 
//Avalanche Technology
//										
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
//										
//	Avalanche Technology Inc., Proprietary and Confidential	   *
//										
// 	Release:  3.14    Date 01/26/2026  	
//							
//////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////
//  PART DESCRIPTION:
//
//  Technology: 22nm pMTJ STT-MRAM
//  Part:       AS301GB32/AS302GB32/AS304GB32/AS308GB32
//
//  Description: 1 Gigabit/2 Gigabit/4 Gigabit/8 Gigabit PARALLEL PERSISTENT SRAM MEMORY
//  Datasheet Revision: REV Z.3
//
////////////////////////////////////////////////////////////////////////////////////
//  FILE CONTENTS : MODEL OF PPSRAM 
//
////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////
// DENSITY SELECTION
//
// Uncomment one of the following lines to select the appropriate density. If
// no line is selected, the model will default to 1Gb
//
// For 1Gb density: ADDR[24:0] with unused ADDR[26:25] connected to GND
// For 2Gb density: ADDR[25:0] with unused ADDR[26] connected to GND
//
////////////////////////////////////////////////////////////////////////////////////
//`define DENSITY_1G
//`define DENSITY_2G
//`define DENSITY_4G
//`define DENSITY_8G

////////////////////////////////////////////////////////////////////////////////////
// MODULE DECLARATION                                                             //
////////////////////////////////////////////////////////////////////////////////////

`timescale 1ns/1ps

`ifdef DENSITY_1G
`define ADDR_WID 27
`define ARRAY_BYTES 32'h2000000
`elsif DENSITY_2G
`define ADDR_WID 27
`define ARRAY_BYTES 32'h4000000
`elsif DENSITY_4G
`define ADDR_WID 27
`define ARRAY_BYTES 32'h8000000
`elsif DENSITY_8G
`define ADDR_WID 27
`define ARRAY_BYTES 32'h8000000
`else
`define ADDR_WID 27
`define ARRAY_BYTES 32'h2000000
`endif

`ifdef DENSITY_8G
module ppsram_as308gb32(
  input                   E1n,
  input                   E2n,
  input                   Wn,
  input [`ADDR_WID-1 : 0] ADDR,
  inout [31 : 0]          DQ,
  input                   Gn,
  input                   PGn,
  input                   CRn,
  output                  INTn
  );
  
  ppsram_as30xgb32 I0(
   .En   (E1n),
   .Wn   (Wn),
   .ADDR (ADDR),
   .DQ   (DQ),
   .Gn   (Gn),
   .PGn  (PGn),
   .CRn  (CRn),
   .INTn (INTn)
   );

  ppsram_as30xgb32 I1(
   .En   (E2n),
   .Wn   (Wn),
   .ADDR (ADDR),
   .DQ   (DQ),
   .Gn   (Gn),
   .PGn  (PGn),
   .CRn  (CRn),
   .INTn (INTn)
   );

endmodule
`endif

module ppsram_as30xgb32(
  input                   En,
  input                   Wn,
  input [`ADDR_WID-1 : 0] ADDR,
  inout [31 : 0]          DQ,
  input                   Gn,
  input                   PGn,
  input                   CRn,
  output                  INTn
  );

  /* Memory Array */
  reg [31:0] mem_array [0 : `ARRAY_BYTES-1];
  reg mem_wpen;

  /* Timing Parameters */
  specify
    specparam tWLEH = 25;
    specparam tWLWH = 25;
    specparam tAVAV = 45;
    specparam tAVWL = 0;
    specparam tDVWH = 15;
    specparam tWHAX = 12;
    specparam tWHDX = 0;
    specparam tAVWH_GnH = 28;
    specparam tAVWH_GnL = 30;

    specparam tWLQZ = 15;
    specparam tWHQX = 3;

    specparam tAVQV = 45;
    specparam tELQV = 45;
    specparam tGLQV = 25;

    specparam tELQX = 3;
    specparam tGLQX = 0;
    specparam tAXQX = 3;

    specparam tEHQZ = 15;
    specparam tGHQZ = 15;

    specparam tPGSL = 10;
    specparam tEH   = 10;
    specparam tPGSH = 10;
    specparam tOE   = 15;
    specparam tAA   = 45;
`ifdef DENSITY_8G
    specparam tAAP  = 20;
`else
    specparam tAAP  = 15;
`endif
    specparam tOHP  = 5;
    specparam tDSP  = 7.5;
    specparam tDHP  = 6;
    specparam tWA   = 45;
    specparam tWP   = 7.5;
    specparam tPWC  = 15;
    specparam tASP  = 7.5;
    specparam tAHP  = 6;
    specparam tdummy = 7.5;
  endspecify

  /* Config mode addr seq */
  localparam CFG_SEQ = {27'h33, 27'h0C, 27'h4A};

  /* Config Registers */
  localparam DEF_ECC_TE = 32'h0;
  localparam DEF_ECC_EI = 32'h0;
  localparam DEF_ECC_DI = 32'h0;
  localparam DEF_ECC_DO = 32'h0;
  localparam DEF_ECC_CR = 32'h0;
  localparam DEF_ECC_EC = 32'h0;
  localparam DEF_IFR    = 32'h0;
  localparam DEF_ODSR   = 32'h0;
  localparam DEF_AWPR   = 32'h0;
  localparam WRMASK_ECC_TE = 32'h00000001;
  localparam WRMASK_ECC_EI = 32'hFFFFFFFF;
  localparam WRMASK_ECC_DI = 32'hFFFFFFFF;
  localparam WRMASK_ECC_DO = 32'h00000000;
  localparam WRMASK_ECC_CR = 32'h00000003;
  localparam WRMASK_ECC_EC = 32'h00000000;
  localparam WRMASK_IFR    = 32'h00000000;
  localparam WRMASK_ODSR   = 32'h00000007;
  localparam WRMASK_AWPR   = 32'h00000007;

  //DIE 1
  reg [31:0] ECC_TE_D1; // ECC Test Enable
  reg [31:0] ECC_EI_D1; // ECC Error Inject
  reg [31:0] ECC_DI_D1; // ECC Data In
  reg [31:0] ECC_DO_D1; // ECC Data Out
  reg [31:0] ECC_CR_D1; // ECC Control Register
  reg [31:0] ECC_EC_D1; // ECC Error Count
  reg [31:0] IFR_D1;    // Interrupt Flag
  reg [31:0] ODSR_D1;   // Output Drive Strength
  reg [31:0] AWPR_D1;   // Array Write Protection
  localparam ADDR_ECC_TE_D1 = 27'h10;
  localparam ADDR_ECC_EI_D1 = 27'h11;
  localparam ADDR_ECC_DI_D1 = 27'h12;
  localparam ADDR_ECC_DO_D1 = 27'h13;
  localparam ADDR_ECC_CR_D1 = 27'h20;
  localparam ADDR_ECC_EC_D1 = 27'h21;
  localparam ADDR_IFR_D1    = 27'h22;
  localparam ADDR_ODSR_D1   = 27'h30;
  localparam ADDR_AWPR_D1   = 27'h40;


  //DIE 2
  reg [31:0] ECC_TE_D2; // ECC Test Enable
  reg [31:0] ECC_EI_D2; // ECC Error Inject
  reg [31:0] ECC_DI_D2; // ECC Data In
  reg [31:0] ECC_DO_D2; // ECC Data Out
  reg [31:0] ECC_CR_D2; // ECC Control Register
  reg [31:0] ECC_EC_D2; // ECC Error Count
  reg [31:0] IFR_D2;    // Interrupt Flag
  reg [31:0] ODSR_D2;   // Output Drive Strength
  reg [31:0] AWPR_D2;   // Array Write Protection
  localparam ADDR_ECC_TE_D2 = 27'h2000010;
  localparam ADDR_ECC_EI_D2 = 27'h2000011;
  localparam ADDR_ECC_DI_D2 = 27'h2000012;
  localparam ADDR_ECC_DO_D2 = 27'h2000013;
  localparam ADDR_ECC_CR_D2 = 27'h2000020;
  localparam ADDR_ECC_EC_D2 = 27'h2000021;
  localparam ADDR_IFR_D2    = 27'h2000022;
  localparam ADDR_ODSR_D2   = 27'h2000030;
  localparam ADDR_AWPR_D2   = 27'h2000040;

  //DIE 3
  reg [31:0] ECC_TE_D3; // ECC Test Enable
  reg [31:0] ECC_EI_D3; // ECC Error Inject
  reg [31:0] ECC_DI_D3; // ECC Data In
  reg [31:0] ECC_DO_D3; // ECC Data Out
  reg [31:0] ECC_CR_D3; // ECC Control Register
  reg [31:0] ECC_EC_D3; // ECC Error Count
  reg [31:0] IFR_D3;    // Interrupt Flag
  reg [31:0] ODSR_D3;   // Output Drive Strength
  reg [31:0] AWPR_D3;   // Array Write Protection
  localparam ADDR_ECC_TE_D3 = 27'h4000010;
  localparam ADDR_ECC_EI_D3 = 27'h4000011;
  localparam ADDR_ECC_DI_D3 = 27'h4000012;
  localparam ADDR_ECC_DO_D3 = 27'h4000013;
  localparam ADDR_ECC_CR_D3 = 27'h4000020;
  localparam ADDR_ECC_EC_D3 = 27'h4000021;
  localparam ADDR_IFR_D3    = 27'h4000022;
  localparam ADDR_ODSR_D3   = 27'h4000030;
  localparam ADDR_AWPR_D3   = 27'h4000040;


  //DIE 4
  reg [31:0] ECC_TE_D4; // ECC Test Enable
  reg [31:0] ECC_EI_D4; // ECC Error Inject
  reg [31:0] ECC_DI_D4; // ECC Data In
  reg [31:0] ECC_DO_D4; // ECC Data Out
  reg [31:0] ECC_CR_D4; // ECC Control Register
  reg [31:0] ECC_EC_D4; // ECC Error Count
  reg [31:0] IFR_D4;    // Interrupt Flag
  reg [31:0] ODSR_D4;   // Output Drive Strength
  reg [31:0] AWPR_D4;   // Array Write Protection
  localparam ADDR_ECC_TE_D4 = 27'h6000010;
  localparam ADDR_ECC_EI_D4 = 27'h6000011;
  localparam ADDR_ECC_DI_D4 = 27'h6000012;
  localparam ADDR_ECC_DO_D4 = 27'h6000013;
  localparam ADDR_ECC_CR_D4 = 27'h6000020;
  localparam ADDR_ECC_EC_D4 = 27'h6000021;
  localparam ADDR_IFR_D4    = 27'h6000022;
  localparam ADDR_ODSR_D4   = 27'h6000030;
  localparam ADDR_AWPR_D4   = 27'h6000040;


  assign INTn = ~IFR_D1[0] & ~IFR_D2[0] & ~IFR_D3[0] & ~IFR_D4[0];
  always@(negedge INTn)
    $display("[MODEL] Interrupt Asserted");

  wire [`ADDR_WID-3 : 0] PGADDR;
  assign PGADDR = ADDR[`ADDR_WID-1 : 2];

  /* Output latch and enable */
  reg [31:0] dout;
  reg [31:0] dread; // internal data read
  reg [127:0] dpage; // Page data
  reg dout_en;

  /* States */
  reg [7:0] state;
  reg cfgmode, cfgaccess;
  reg cfgdummy_wr;
  localparam S_IDLE = 8'd0;
  localparam S_ENABLE = 8'd1;
  localparam S_WRWAIT = 8'd2;
  localparam S_WRDONE = 8'd3;
  localparam S_PGIDLE = 8'd4;
  localparam S_PGENA  = 8'd5;
  localparam S_PGWRWA = 8'd6;
  localparam S_PGWRDN = 8'd7;
  localparam S_PGWRI2 = 8'd9;
  localparam S_PGWRW2 = 8'd10;
  localparam S_PGWRD2 = 8'd11;
  localparam S_PGEXIT = 8'd8;
  localparam S_ERROR  = 8'hFF;

  /* En low counter */
  integer En_lctr;
  always@(negedge En) begin
    En_lctr <= 0;
    while (!En) begin
      #1 En_lctr <= En_lctr + 1;
    end
  end

  /* Wn counter */
  integer Wn_lctr;
  always@(negedge Wn) begin
    Wn_lctr <= 0;
    while (!Wn) begin
      #1 Wn_lctr <= Wn_lctr + 1;
    end
  end

  /* Gn low counter */
  integer Gn_lctr;
  always@(negedge Gn) begin
    Gn_lctr <= 0;
    while (!Gn) begin
      #1 Gn_lctr <= Gn_lctr + 1;
    end
  end

  /* Gn high counter */
  integer Gn_hctr;
  always@(posedge Gn) begin
    Gn_hctr <= 0;
    while (Gn) begin
      #1 Gn_hctr <= Gn_hctr + 1;
    end
  end

  /* En high counter */
  integer En_hctr;

  /* PGn low counter */
  integer PGn_lctr;
  always@(negedge PGn) begin
    PGn_lctr <= 0;
    while (!PGn) begin
      #1 PGn_lctr <= PGn_lctr + 1;
    end
  end

  /* PGn high counter */
  integer PGn_hctr;
  always@(posedge PGn) begin
    PGn_hctr <= 0;
    while (PGn) begin
      #1 PGn_hctr <= PGn_hctr + 1;
    end
  end

  /* Address stable counter */
  integer ADDR_ctr;
  reg ADDR_ch;
  always@(ADDR) begin
    ADDR_ch <= 1'b1;
    #1 ADDR_ch <= 0;
  end
  always@(posedge ADDR_ch or negedge ADDR_ch) begin
    if(ADDR_ch)
      ADDR_ctr = ADDR_ctr + 1;
    else begin
      ADDR_ctr = 0;
      while(!ADDR_ch) begin
        #1;
        ADDR_ctr = ADDR_ctr + 1;
      end
    end
  end

  integer ADDR_ctr_q;
  always@(posedge ADDR_ch) begin
    ADDR_ctr_q = 0;
  end
  always@(negedge ADDR_ch) begin
    ADDR_ctr_q = ADDR_ctr_q + 1;
    #1;
    while(!ADDR_ch) begin
      ADDR_ctr_q = ADDR_ctr_q + 1;
      #1;
    end
  end

  /* Page Address stable counter */
  integer PGADDR_ctr;
  reg PGADDR_ch;
  always@(ADDR[1:0]) begin
    if(ADDR[1:0] == 0) begin
      PGADDR_ch <= 1'b1;
      #1 PGADDR_ch <= 0;
    end
  end
  always@(posedge PGADDR_ch or negedge PGADDR_ch) begin
    if(PGADDR_ch)
      PGADDR_ctr = PGADDR_ctr + 1;
    else begin
      PGADDR_ctr = 0;
      while(!PGADDR_ch) begin
        #1;
        PGADDR_ctr = PGADDR_ctr + 1;
      end
    end
  end

  /* Data stable counter */
  integer din_ctr;
  reg DQ_ch;
  always@(DQ) begin
    DQ_ch <= 1'b1;
    #1 DQ_ch <= 0;
  end
  always@(posedge DQ_ch or negedge DQ_ch) begin
    if(DQ_ch)
      din_ctr = din_ctr + 1;
    else begin
      din_ctr = 0;
      while(!DQ_ch) begin
        #1;
        din_ctr = din_ctr + 1;
      end
    end
  end

  /* Write delay counter */
  integer wrdly_ctr;
  reg wrdly_en;
  always@(state)
    if (state == S_WRDONE) begin
      wrdly_en <= 1;
    end
    else if ((state == S_PGWRDN) || (state == S_PGWRD2)) begin
      wrdly_en <= 1;
    end
    else begin
      wrdly_en <= 0;
    end
  always begin
    if (wrdly_en) begin
      wrdly_ctr <= wrdly_ctr + 1;
      #1;
    end
    else begin
      wrdly_ctr <= 0;
      #1;
    end
  end

  /* Wn negedge counter */
  integer Wn_nectr;
  always@(negedge Wn)
    Wn_nectr = 0;
  always@(ADDR_ctr) 
    Wn_nectr = Wn_nectr + 1;

  /* State Machine */
  always@(posedge En
          or posedge Wn
          or negedge En 
          or negedge Wn
          or negedge PGn
          or En_lctr
          or Wn_lctr
          or wrdly_ctr
          or posedge ADDR_ch
         )
    case(state)
      S_IDLE:
        if (!PGn) begin
          state <= S_PGIDLE;
          cfgmode <= 0;
          cfgaccess <= 0;
        end
        else if (!En) begin
          state <= S_ENABLE;
          if (!CRn) begin
            cfgmode <= 1'b1;
          end
          else begin
            if (cfgmode) begin
              $display("[MODEL] Config register access disabled");
            end
            cfgmode <= 0;
            cfgaccess <= 0;
          end
        end
        else begin
          state <= S_IDLE;
        end

      S_ENABLE:
        if (!Wn) begin
          if (ADDR_ctr >= tAVWL-1) begin
            state <= S_WRWAIT; // continue with write
          end
          else begin
            state <= S_ERROR; // ignore write
            $display("[MODEL] Write ignored. tAVWL violation");
          end
        end
        else if (En) begin
          state <= S_IDLE;
        end
        else if (cfgmode && CRn) begin // CRn deasserted before chip disabled
          state <= S_ERROR;
          $display("[MODEL] Error: CRn deasserted before En");
        end
        else
          state <= S_ENABLE;

      S_WRWAIT:
        if (Wn | En) begin
          if ((Wn_lctr >= tWLWH-1) && (En_lctr >= tWLEH-1)) begin
            if (din_ctr >= tDVWH-2) begin
              if (Gn) begin
                if (ADDR_ctr >= tAVWH_GnH-2) begin
                  state <= S_WRDONE; // continue with write
                end
                else begin
                  state <= S_ERROR; // ignore write
                  $display("[MODEL] Write ignored. tAVWH violation");
                end
              end
              else begin
                if (ADDR_ctr >= tAVWH_GnL-2) begin
                  state <= S_WRDONE; // continue with write
                end
                else begin
                  state <= S_ERROR; // ignore write
                  $display("[MODEL] Write ignored. tAVWH violation");
                end
              end
            end
            else begin
              state <= S_ERROR; // ignore write
              $display("[MODEL] Write ignored. tDVWH violation");
            end
          end
          else begin
            state <= S_ERROR; // ignore write
            $display("[MODEL] Write ignored. tWLWH/tWLEH violation");
          end
        end
        else if (cfgmode && CRn) begin // CRn deasserted in the middle of write
          state <= S_ERROR;
          $display("[MODEL] Error: CRn deasserted in the middle of write");
        end
        else begin
          state <= S_WRWAIT;
        end

      S_WRDONE:
        if (ADDR_ch) begin 
          if (ADDR_ctr < tAVAV-2) begin
            $display("[MODEL] Write Ignored. tAVAV violation");
          end
          if (!En)
            state <= S_ENABLE;
          else
            state <= S_IDLE;
        end
        else if ((ADDR_ctr >= tAVAV-1) && En && Wn && (wrdly_ctr >= tWHAX)) begin
          state <= S_IDLE;
        end
        else begin
          state <= S_WRDONE;
        end

      S_PGIDLE:
        if (PGn) begin
          state <= S_PGEXIT;
        end
        else if (!En) begin
          if (PGn_lctr >= tPGSL-1) begin
            $display("[MODEL] Page mode enabled");
            state <= S_PGENA; // Enable page mode
          end
          else begin
            $display("[MODEL] Page mode select ignored. tPGSL violation");
            state <= S_PGIDLE; //remain in Page mode idle
          end
        end
        else begin
          state <= S_PGIDLE;
        end

      S_PGENA: begin
        if (En || PGn) begin
          state <= S_PGEXIT;
        end
        else if (!Wn) begin
          if (ADDR_ctr >= tAVWL-1) begin
            state <= S_PGWRWA;
          end
          else begin
            state <= S_ERROR; // ignore write
            $display("[MODEL] Write ignored. tAVWL violation");
          end
        end
        else begin
          state <= S_PGENA;
        end
      end

      S_PGWRWA:
        if (PGn | En) begin
          state <= S_PGEXIT;
        end
        else if (Wn) begin
          if ((Wn_lctr >= tWLWH-1) && (En_lctr >= tWLEH-1)) begin
            if (din_ctr >= tDSP-2) begin
              if (Gn) begin
                if (ADDR_ctr >= tAVWH_GnH-2) begin
                  state <= S_PGWRDN; // continue with write
                end
                else begin
                  state <= S_ERROR; // ignore write
                  $display("[MODEL] Write ignored. tAVWH violation");
                end
              end
              else begin
                if (ADDR_ctr >= tAVWH_GnL-2) begin
                  state <= S_PGWRDN; // continue with write
                end
                else begin
                  state <= S_ERROR; // ignore write
                  $display("[MODEL] Write ignored. tAVWH violation");
                end
              end
            end
            else begin
              state <= S_ERROR; // ignore write
              $display("[MODEL] Write ignored. tDSP violation");
            end
          end
          else begin
            state <= S_ERROR; // ignore write
            $display("[MODEL] Write ignored. tWLWH/tWLEH violation");
          end
        end
        else begin
          state <= S_PGWRWA;
        end

      S_PGWRDN: begin
        if (PGn | En) begin
          state <= S_PGEXIT;
        end
        else if ((Wn_lctr + wrdly_ctr) >= tWA-1
           && (wrdly_ctr >= tDHP-1)) begin
          state <= S_PGWRI2;
        end
        /*if ((Wn_lctr + wrdly_ctr) >= tWA) begin
          if (!Wn)
            state <= S_PGWRW2;
          else
            state <= S_PGWRI2;
        end*/
        else if (!Wn) begin
          $display("[MODEL] Page Write ignored. tPWC violation");
          state <= S_ERROR;
        end
        else begin
          state <= S_PGWRDN;
        end
      end

      S_PGWRI2:
        if (PGn | En) begin
          state <= S_PGEXIT;
        end
        else if (!Wn)
          state <= S_PGWRW2;
        else
          state <= S_PGWRI2;

      S_PGWRW2:
        if (PGn | En) begin
          state <= S_PGEXIT;
        end
        else if (Wn) begin
          if (Wn_lctr >= tWP-1) begin
            if (din_ctr >= tDSP-2) begin
              state <= S_PGWRD2;
            end
            else begin
              state <= S_ERROR; // ignore write
              $display("[MODEL] Page Write ignored. tDSP violation");
            end
          end
          else begin
            state <= S_ERROR; // ignore write
            $display("[MODEL] Page Write ignored. tWP violation");
          end
        end
        else begin
          state <= S_PGWRW2;
        end

      S_PGWRD2: begin
        if (PGn | En) begin
          state <= S_PGEXIT;
        end
        else if ((Wn_lctr + wrdly_ctr) >= tPWC
           && (wrdly_ctr >= tDHP-1)
           && (ADDR_ctr >= tASP-1)
           ) begin
          state <= S_PGWRI2;
        end
        else if (!Wn) begin
          state <= S_ERROR;
	  if((Wn_lctr + wrdly_ctr) < tPWC)
            $display("[MODEL] Page Write ignored. tPWC violation");
	  if( wrdly_ctr < tDHP-1)
            $display("[MODEL] Page Write ignored. tDHP violation");
        end
        /*if ((Wn_lctr + wrdly_ctr) > tPWC) begin
          if (!Wn)
            state <= S_PGWRW2;
          else
            state <= S_PGWRI2;
        end*/
        else begin
          state <= S_PGWRD2;
        end
      end

      S_PGEXIT:
        if ((En_hctr >= tEH)
            && (PGn_hctr >= tPGSH-1)
           ) begin
          state <= S_IDLE;
          $display("[MODEL] Page mode disabled");
        end
        else if (!En) begin
          state <= S_PGEXIT;
          $display("[MODEL] Still exiting Page Mode. tEH/tPGSH violation");
        end
        else begin
          state <= S_PGEXIT;
        end

      S_ERROR:
        if (En)
          state <= S_IDLE;
        else
          state <= S_ERROR;

      default:
        state <= S_ERROR;
    endcase

    /* Data in latch */
    reg [31:0] din;
    always@(posedge Wn or posedge En)
      din <= DQ;

    /* Write trigger */
    reg wr_trig;
    always@(wrdly_ctr or state)
      if ((state == S_WRDONE) && (wrdly_ctr >= tWHAX-1)) begin
        if (ADDR_ctr >= tAVAV-2) begin
          wr_trig <= 1'b1;
        end
        else begin
          wr_trig <= 0;
          //$display("[MODEL] Write ignored. tAVAV violation");
        end
      end
      else if ((state == S_PGWRDN) && (wrdly_ctr >= tDHP-1)) begin
        if (ADDR_ctr >= tWA-2) begin
          wr_trig <= 1'b1;
        end
        else begin
          wr_trig <= 0;
        end
      end
      else if ((state == S_PGWRI2) && (wrdly_ctr >= tDHP-1)) begin
        if (ADDR_ctr >= (tASP+tAHP-1)) begin
          wr_trig <= 1'b1;
        end
        else begin
          wr_trig <= 0;
        end
      end
      else begin
        wr_trig <= 0;
      end

    reg [(`ADDR_WID*2)-1 : 0] cfgdummy_addr;
    always@(posedge wr_trig) begin
      if (cfgmode) begin
        if (!cfgaccess) begin // Config reg not yet accessible
          if ({cfgdummy_addr,ADDR} == CFG_SEQ) begin // Allow access
            cfgaccess = 1'b1;
            $display("[MODEL] Config register access enabled");
          end
          else begin
            cfgaccess = 0;
            cfgdummy_addr[`ADDR_WID*2-1 : `ADDR_WID] = cfgdummy_addr[`ADDR_WID-1 : 0];
            cfgdummy_addr[`ADDR_WID-1 : 0] =  ADDR;
          end
        end

        else begin // Config reg accessible
          $display("[MODEL] Config Register 0x%X written with 0x%X",ADDR,din);
          case(ADDR)
            ADDR_ECC_TE_D1: ECC_TE_D1 = (ECC_TE_D1 & ~WRMASK_ECC_TE) | (din & WRMASK_ECC_TE);
            ADDR_ECC_EI_D1: ECC_EI_D1 = (ECC_EI_D1 & ~WRMASK_ECC_EI) | (din & WRMASK_ECC_EI);
            ADDR_ECC_DI_D1: ECC_DI_D1 = (ECC_DI_D1 & ~WRMASK_ECC_DI) | (din & WRMASK_ECC_DI);
            ADDR_ECC_DO_D1: ECC_DO_D1 = (ECC_DO_D1 & ~WRMASK_ECC_DO) | (din & WRMASK_ECC_DO);
            ADDR_ECC_CR_D1: begin
              ECC_CR_D1 = (ECC_CR_D1 & ~WRMASK_ECC_CR) | (din & WRMASK_ECC_CR);
              if (ECC_CR_D1[1]) begin // Interrupt reset
                IFR_D1[0] = 0;
                ECC_CR_D1[1] = 0; // toggle back to 0
              end
              if (ECC_CR_D1[0]) begin // Clear error count
                ECC_EC_D1 = 0;
                ECC_CR_D1[0] = 0; // toggle back to 0
              end
            end
            ADDR_ECC_EC_D1: ECC_EC_D1 = (ECC_EC_D1 & ~WRMASK_ECC_EC) | (din & WRMASK_ECC_EC);
            ADDR_IFR_D1   : IFR_D1    = (IFR_D1    & ~WRMASK_IFR)    | (din & WRMASK_IFR);
            ADDR_ODSR_D1  : ODSR_D1   = (ODSR_D1   & ~WRMASK_ODSR)   | (din & WRMASK_ODSR);
            ADDR_AWPR_D1  : AWPR_D1   = (AWPR_D1   & ~WRMASK_AWPR)   | (din & WRMASK_AWPR);

            ADDR_ECC_TE_D2: ECC_TE_D2 = (ECC_TE_D2 & ~WRMASK_ECC_TE) | (din & WRMASK_ECC_TE);
            ADDR_ECC_EI_D2: ECC_EI_D2 = (ECC_EI_D2 & ~WRMASK_ECC_EI) | (din & WRMASK_ECC_EI);
            ADDR_ECC_DI_D2: ECC_DI_D2 = (ECC_DI_D2 & ~WRMASK_ECC_DI) | (din & WRMASK_ECC_DI);
            ADDR_ECC_DO_D2: ECC_DO_D2 = (ECC_DO_D2 & ~WRMASK_ECC_DO) | (din & WRMASK_ECC_DO);
            ADDR_ECC_CR_D2: begin
              ECC_CR_D2 = (ECC_CR_D2 & ~WRMASK_ECC_CR) | (din & WRMASK_ECC_CR);
              if (ECC_CR_D2[1]) begin // Interrupt reset
                IFR_D2[0] = 0;
                ECC_CR_D2[1] = 0; // toggle back to 0
              end
              if (ECC_CR_D2[0]) begin // Clear error count
                ECC_EC_D2 = 0;
                ECC_CR_D2[0] = 0; // toggle back to 0
              end
            end
            ADDR_ECC_EC_D2: ECC_EC_D2 = (ECC_EC_D2 & ~WRMASK_ECC_EC) | (din & WRMASK_ECC_EC);
            ADDR_IFR_D2   : IFR_D2    = (IFR_D2    & ~WRMASK_IFR)    | (din & WRMASK_IFR);
            ADDR_ODSR_D2  : ODSR_D2   = (ODSR_D2   & ~WRMASK_ODSR)   | (din & WRMASK_ODSR);
            ADDR_AWPR_D2  : AWPR_D2   = (AWPR_D2   & ~WRMASK_AWPR)   | (din & WRMASK_AWPR);

            ADDR_ECC_TE_D3: ECC_TE_D3 = (ECC_TE_D3 & ~WRMASK_ECC_TE) | (din & WRMASK_ECC_TE);
            ADDR_ECC_EI_D3: ECC_EI_D3 = (ECC_EI_D3 & ~WRMASK_ECC_EI) | (din & WRMASK_ECC_EI);
            ADDR_ECC_DI_D3: ECC_DI_D3 = (ECC_DI_D3 & ~WRMASK_ECC_DI) | (din & WRMASK_ECC_DI);
            ADDR_ECC_DO_D3: ECC_DO_D3 = (ECC_DO_D3 & ~WRMASK_ECC_DO) | (din & WRMASK_ECC_DO);
            ADDR_ECC_CR_D3: begin
              ECC_CR_D3 = (ECC_CR_D3 & ~WRMASK_ECC_CR) | (din & WRMASK_ECC_CR);
              if (ECC_CR_D3[1]) begin // Interrupt reset
                IFR_D3[0] = 0;
                ECC_CR_D3[1] = 0; // toggle back to 0
              end
              if (ECC_CR_D3[0]) begin // Clear error count
                ECC_EC_D3 = 0;
                ECC_CR_D3[0] = 0; // toggle back to 0
              end
            end
            ADDR_ECC_EC_D3: ECC_EC_D3 = (ECC_EC_D3 & ~WRMASK_ECC_EC) | (din & WRMASK_ECC_EC);
            ADDR_IFR_D3   : IFR_D3    = (IFR_D3    & ~WRMASK_IFR)    | (din & WRMASK_IFR);
            ADDR_ODSR_D3  : ODSR_D3   = (ODSR_D3   & ~WRMASK_ODSR)   | (din & WRMASK_ODSR);
            ADDR_AWPR_D3  : AWPR_D3   = (AWPR_D3   & ~WRMASK_AWPR)   | (din & WRMASK_AWPR);

            ADDR_ECC_TE_D4: ECC_TE_D4 = (ECC_TE_D4 & ~WRMASK_ECC_TE) | (din & WRMASK_ECC_TE);
            ADDR_ECC_EI_D4: ECC_EI_D4 = (ECC_EI_D4 & ~WRMASK_ECC_EI) | (din & WRMASK_ECC_EI);
            ADDR_ECC_DI_D4: ECC_DI_D4 = (ECC_DI_D4 & ~WRMASK_ECC_DI) | (din & WRMASK_ECC_DI);
            ADDR_ECC_DO_D4: ECC_DO_D4 = (ECC_DO_D4 & ~WRMASK_ECC_DO) | (din & WRMASK_ECC_DO);
            ADDR_ECC_CR_D4: begin
              ECC_CR_D4 = (ECC_CR_D4 & ~WRMASK_ECC_CR) | (din & WRMASK_ECC_CR);
              if (ECC_CR_D4[1]) begin // Interrupt reset
                IFR_D4[0] = 0;
                ECC_CR_D4[1] = 0; // toggle back to 0
              end
              if (ECC_CR_D4[0]) begin // Clear error count
                ECC_EC_D4 = 0;
                ECC_CR_D4[0] = 0; // toggle back to 0
              end
            end
            ADDR_ECC_EC_D4: ECC_EC_D4 = (ECC_EC_D4 & ~WRMASK_ECC_EC) | (din & WRMASK_ECC_EC);
            ADDR_IFR_D4   : IFR_D4    = (IFR_D4    & ~WRMASK_IFR)    | (din & WRMASK_IFR);
            ADDR_ODSR_D4  : ODSR_D4   = (ODSR_D4   & ~WRMASK_ODSR)   | (din & WRMASK_ODSR);
            ADDR_AWPR_D4  : AWPR_D4   = (AWPR_D4   & ~WRMASK_AWPR)   | (din & WRMASK_AWPR);

            default: begin // Trigger dummy write
              $display("[MODEL] Config Register dummy write");
              cfgdummy_wr <= 1'b1;
              #1 cfgdummy_wr <= 0;
            end
          endcase
        end
      end

      else begin
        if (mem_wpen) begin
          $display("[MODEL] Write ignored. Addr 0x%X is protected", ADDR);
        end
        else begin
	  if(!PGn) begin
            mem_array[ADDR] = DQ;
            $display("[MODEL] Data 0x%X written to Addr 0x%X", DQ, ADDR);
	  end
	  else begin
            mem_array[ADDR] = din;
            $display("[MODEL] Data 0x%X written to Addr 0x%X", din, ADDR);
	  end
        end
      end
    end

    /* Internal read */
    always@(negedge Gn or negedge En or negedge ADDR_ch) begin
      if (!En && !Gn) begin
        if (!CRn) begin
          case(ADDR)
            ADDR_ECC_TE_D1: dread = ECC_TE_D1;
            ADDR_ECC_EI_D1: dread = ECC_EI_D1;
            ADDR_ECC_DI_D1: dread = ECC_DI_D1;
            ADDR_ECC_DO_D1: dread = ECC_DO_D1;
            ADDR_ECC_CR_D1: dread = ECC_CR_D1;
            ADDR_ECC_EC_D1: dread = ECC_EC_D1;
            ADDR_IFR_D1   : dread = IFR_D1;
            ADDR_ODSR_D1  : dread = ODSR_D1;
            ADDR_AWPR_D1  : dread = AWPR_D1;

            ADDR_ECC_TE_D2: dread = ECC_TE_D2;
            ADDR_ECC_EI_D2: dread = ECC_EI_D2;
            ADDR_ECC_DI_D2: dread = ECC_DI_D2;
            ADDR_ECC_DO_D2: dread = ECC_DO_D2;
            ADDR_ECC_CR_D2: dread = ECC_CR_D2;
            ADDR_ECC_EC_D2: dread = ECC_EC_D2;
            ADDR_IFR_D2   : dread = IFR_D2;
            ADDR_ODSR_D2  : dread = ODSR_D2;
            ADDR_AWPR_D2  : dread = AWPR_D2;

            ADDR_ECC_TE_D3: dread = ECC_TE_D3;
            ADDR_ECC_EI_D3: dread = ECC_EI_D3;
            ADDR_ECC_DI_D3: dread = ECC_DI_D3;
            ADDR_ECC_DO_D3: dread = ECC_DO_D3;
            ADDR_ECC_CR_D3: dread = ECC_CR_D3;
            ADDR_ECC_EC_D3: dread = ECC_EC_D3;
            ADDR_IFR_D3   : dread = IFR_D3;
            ADDR_ODSR_D3  : dread = ODSR_D3;
            ADDR_AWPR_D3  : dread = AWPR_D3;

            ADDR_ECC_TE_D4: dread = ECC_TE_D4;
            ADDR_ECC_EI_D4: dread = ECC_EI_D4;
            ADDR_ECC_DI_D4: dread = ECC_DI_D4;
            ADDR_ECC_DO_D4: dread = ECC_DO_D4;
            ADDR_ECC_CR_D4: dread = ECC_CR_D4;
            ADDR_ECC_EC_D4: dread = ECC_EC_D4;
            ADDR_IFR_D4   : dread = IFR_D4;
            ADDR_ODSR_D4  : dread = ODSR_D4;
            ADDR_AWPR_D4  : dread = AWPR_D4;
            default: begin 
              $display("[MODEL] Invalid Config Register Address");
              dread = 32'h0;
            end
          endcase
        end

        else begin
          dread <= mem_array[ADDR];
        end
      end
    end

    /* Page internal read */
    always@(negedge Gn
           or posedge ADDR_ch
           ) begin
      if (!PGn) begin
        dpage[127:96] = mem_array[{PGADDR, 2'b00}];
        dpage[95:64]  = mem_array[{PGADDR, 2'b01}];
        dpage[63:32]  = mem_array[{PGADDR, 2'b10}];
        dpage[31:0]   = mem_array[{PGADDR, 2'b11}];
      end
    end

    /* Page word latch */
    reg [1:0] addr_pword;
    always@(posedge ADDR_ch) begin
      addr_pword = ADDR[1:0];
    end

    reg [31:0] pgdread;
    reg pgdread_trig;
    always@(negedge Gn
            or posedge pgdread_trig
           ) begin
      case(addr_pword)
        2'b00: pgdread = dpage[127:96];
        2'b01: pgdread = dpage[95:64];
        2'b10: pgdread = dpage[63:32];
        2'b11: pgdread = dpage[31:0];
      endcase
    end

    always@(ADDR_ctr)
      if (ADDR_ctr == tAAP-1)
        pgdread_trig = 1'b1;
      else
        pgdread_trig = 0;

    /* Output latch */
    always@(Gn_lctr)
      if (((En_lctr >= tELQV)
          || (Gn_lctr >= tGLQV)
          || (ADDR_ctr >= tAVQV-1))
          && (state != S_PGENA)
	  && (ADDR_ctr >= tAXQX-1)
         ) begin
        dout = dread;
      end

    /* Page read completed flag */
    reg pgread_completed;
    reg pgread_prevcomp;
    always@(posedge ADDR_ch)
      pgread_prevcomp = pgread_completed;

    /* Output enable state machine */
    localparam S_OEIDLE = 8'd0;
    localparam S_OEWAIT = 8'd1;
    localparam S_OEACTV = 8'd2;
    localparam S_OEDELY = 8'd3;
    localparam S_OEPGIDLE = 8'd4;
    localparam S_OEPGACTV = 8'd5;
    localparam S_OEPGDELY = 8'd6;
    localparam S_OEPGIDLE2 = 8'd7;
    localparam S_OEERROR = 8'hff;
    reg [7:0] oestate;
    always@(En_lctr
            or En_hctr
            or Gn_lctr
            or Gn_hctr
           )begin
      case(oestate)
        S_OEIDLE: begin
          if (state == S_PGENA) begin
            oestate <= S_OEPGIDLE;
          end
          else if (!En && !Gn && Wn) begin
            if ((En_lctr >= tELQX-1)
                //&& (Gn_lctr >= tGLQX-1) // Comment out if tGLQX = 0
               ) begin
              oestate <= S_OEACTV;
            end
            else begin
              oestate <= S_OEWAIT;
            end
          end
          else begin
            oestate <= S_OEIDLE;
          end
        end

        S_OEWAIT: begin
          if (En | Gn | !Wn) begin
            oestate <= S_OEIDLE;
          end
          else if ((En_lctr >= tELQX-1)
                   //&& (Gn_lctr >= tGLQX-1) // Comment out if tGLQX = 0
                  ) begin
            oestate <= S_OEACTV;
          end
          else begin
            oestate <= S_OEWAIT;
          end
        end

        S_OEACTV: begin
          if (!Wn) begin
            oestate <= S_OEIDLE;
          end
          else if (ADDR_ch) begin 
           if (ADDR_ctr < tAVAV-1) begin
            $display("[MODEL] Read Ignored. tAVAV violation");
            oestate <= S_OEERROR;
           end
           if (!En) begin
            if ((En_lctr >= tELQX-1)
                //&& (Gn_lctr >= tGLQX-1) // Comment out if tGLQX = 0
               ) begin
              oestate <= S_OEACTV;
            end
            else begin
            oestate <= S_OEWAIT;
	    end
	   end
           else
            oestate <= S_OEIDLE;
          end
          else if (En | Gn) begin
            oestate <= S_OEDELY;
          end
          else begin
            oestate <= S_OEACTV;
          end
        end

        S_OEDELY: begin
          if (ADDR_ch) begin 
           if (ADDR_ctr < tAVAV-1) begin
            $display("[MODEL] Read Ignored. tAVAV violation");
            oestate <= S_OEERROR;
           end
           if (!En) begin
            if ((En_lctr >= tELQX-1)
                //&& (Gn_lctr >= tGLQX-1) // Comment out if tGLQX = 0
               ) begin
              oestate <= S_OEACTV;
            end
            else begin
            oestate <= S_OEWAIT;
	    end
	   end
           else
            oestate <= S_OEIDLE;
          end
          else if ((En_hctr >= tEHQZ)
              && (Gn_hctr >= tGHQZ)
             ) begin

            if (!En && !Gn) begin
              if ((En_lctr >= tELQX-1)
                  //&& (Gn_lctr >= tGLQX-1) // Comment out if tGLQX = 0;
                 ) begin
                oestate <= S_OEACTV;
              end
              else begin
                oestate <= S_OEWAIT;
              end
            end
            else begin
              oestate <= S_OEIDLE;
            end

          end
          else begin
            oestate <= S_OEDELY;
          end
        end

        S_OEPGIDLE:
          if (!Gn) begin
            if ((Gn_lctr >= tOE)
                && (ADDR_ctr >= tAA-1)
               ) begin
              oestate <= S_OEPGACTV;
              pgread_completed <= 1'b1;
            end
            else begin
              oestate <= S_OEPGIDLE;
            end
          end
          else if (state != S_PGENA) begin
            oestate <= S_OEIDLE;
          end
          else begin
            oestate <= S_OEPGIDLE;
          end 

        S_OEPGACTV:
          if (ADDR_ch || (ADDR_ctr == 0)) begin
            pgread_completed <= 0;
            if (PGADDR_ch || (PGADDR_ctr == 0)) begin // first access again
              oestate <= S_OEPGDELY;
              $display("[MODEL] Page address changed");
            end
            else begin // same page access
              oestate <= S_OEPGDELY;
            end
          end
          else if (state != S_PGENA) begin
            oestate <= S_OEIDLE;
          end
          else begin
            oestate <= S_OEPGACTV;
          end

        S_OEPGDELY:
          if (ADDR_ctr >= tAAP-2) begin
            oestate <= S_OEPGACTV;
            pgread_completed <= 1'b1;
          end
          else if (state != S_PGENA) begin
            oestate <= S_OEIDLE;
          end
          else begin
            oestate <= S_OEPGDELY;
          end

        S_OEPGIDLE2:
          if (!Gn) begin
            if ((Gn_lctr >= tOE)
                && (ADDR_ctr >= tAA-1)
               ) begin
              oestate <= S_OEPGACTV;
              pgread_completed <= 1'b1;
            end
            else begin
              oestate <= S_OEPGIDLE2;
            end
          end
          else if (state != S_PGENA) begin
            oestate <= S_OEIDLE;
          end
          else begin
            oestate <= S_OEPGIDLE2;
          end 

      S_OEERROR:
        if (En)
          oestate <= S_OEIDLE;
        else
          oestate <= S_OEERROR;

        default: begin
          oestate <= S_OEIDLE;
        end
      endcase
    end

    always@(Gn_hctr or PGn_hctr) begin
      if (oestate == S_OEDELY) begin
        En_hctr <= En_hctr + 1;
      end
      else if (state == S_PGEXIT) begin
        En_hctr <= En_hctr + 1;
      end
      else begin
        En_hctr <= 0;
      end
    end


    /* Output enable */
    always@(*)
      case(oestate)
        S_OEACTV, S_OEDELY: dout_en = 1'b1;
        S_OEPGACTV: dout_en = 1'b1;
        S_OEPGDELY, S_OEPGIDLE2:
          if (pgread_prevcomp && (ADDR_ctr_q <= tOHP-1))
            dout_en = 1'b1;
          else
            dout_en = 0;
        default: dout_en = 0;
      endcase

  assign DQ = (dout_en)? ((state == S_PGENA)? pgdread : dout) : 32'hz;

    /* Array write protection */
    localparam UPPER_1_64 = 25'h1F80000;
    localparam UPPER_1_32 = 25'h1F00000;
    localparam UPPER_1_16 = 25'h1E00000;
    localparam UPPER_1_8  = 25'h1C00000;
    localparam UPPER_1_4  = 25'h1800000;
    localparam UPPER_1_2  = 25'h1000000;

    always@(*) begin
     if(ADDR[26:25] == 2'b00) begin
      case(AWPR_D1[2:0])
        3'b000:
          mem_wpen <= 0;
        3'b001:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_64)? 1'b1 : 0;
        3'b010:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_32)? 1'b1 : 0;
        3'b011:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_16)? 1'b1 : 0;
        3'b100:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_8)? 1'b1 : 0;
        3'b101:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_4)? 1'b1 : 0;
        3'b110:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_2)? 1'b1 : 0;
        3'b111:
          mem_wpen <= 1'b1;
      endcase
     end
     else if(ADDR[26:25] == 2'b01) begin
      case(AWPR_D2[2:0])
        3'b000:
          mem_wpen <= 0;
        3'b001:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_64)? 1'b1 : 0;
        3'b010:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_32)? 1'b1 : 0;
        3'b011:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_16)? 1'b1 : 0;
        3'b100:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_8)? 1'b1 : 0;
        3'b101:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_4)? 1'b1 : 0;
        3'b110:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_2)? 1'b1 : 0;
        3'b111:
          mem_wpen <= 1'b1;
      endcase
     end
     else if(ADDR[26:25] == 2'b10) begin
      case(AWPR_D3[2:0])
        3'b000:
          mem_wpen <= 0;
        3'b001:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_64)? 1'b1 : 0;
        3'b010:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_32)? 1'b1 : 0;
        3'b011:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_16)? 1'b1 : 0;
        3'b100:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_8)? 1'b1 : 0;
        3'b101:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_4)? 1'b1 : 0;
        3'b110:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_2)? 1'b1 : 0;
        3'b111:
          mem_wpen <= 1'b1;
      endcase
     end
     else if(ADDR[26:25] == 2'b11) begin
      case(AWPR_D4[2:0])
        3'b000:
          mem_wpen <= 0;
        3'b001:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_64)? 1'b1 : 0;
        3'b010:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_32)? 1'b1 : 0;
        3'b011:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_16)? 1'b1 : 0;
        3'b100:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_8)? 1'b1 : 0;
        3'b101:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_4)? 1'b1 : 0;
        3'b110:
          mem_wpen <= (ADDR[24:0] >= UPPER_1_2)? 1'b1 : 0;
        3'b111:
          mem_wpen <= 1'b1;
      endcase
     end
     else begin
	mem_wpen <= 0;
     end
    end

    /* ECC Engine
     * - In this model, no error correction applied
     * - Triggered by ECC Test Enable Register and a dummy write
     */
    integer i;
    always@(posedge cfgdummy_wr) begin
      if (ECC_TE_D1[0]) begin // Test Enabled
        ECC_DO_D1 = ECC_DI_D1 ^ ECC_EI_D1;
        IFR_D1 = 32'h1; // Always triggers since no ECC

        for (i=0; i<32; i=i+1) begin
          if (ECC_EI_D1[i]) begin
            ECC_EC_D1 = ECC_EC_D1 + 1;
          end
        end
      end

      if (ECC_TE_D2[0]) begin // Test Enabled
        ECC_DO_D2 = ECC_DI_D2 ^ ECC_EI_D2;
        IFR_D2 = 32'h1; // Always triggers since no ECC

        for (i=0; i<32; i=i+1) begin
          if (ECC_EI_D2[i]) begin
            ECC_EC_D2 = ECC_EC_D2 + 1;
          end
        end
      end

      if (ECC_TE_D3[0]) begin // Test Enabled
        ECC_DO_D3 = ECC_DI_D3 ^ ECC_EI_D3;
        IFR_D3 = 32'h1; // Always triggers since no ECC

        for (i=0; i<32; i=i+1) begin
          if (ECC_EI_D3[i]) begin
            ECC_EC_D3 = ECC_EC_D3 + 1;
          end
        end
      end

      if (ECC_TE_D4[0]) begin // Test Enabled
        ECC_DO_D4 = ECC_DI_D4 ^ ECC_EI_D4;
        IFR_D4 = 32'h1; // Always triggers since no ECC

        for (i=0; i<32; i=i+1) begin
          if (ECC_EI_D4[i]) begin
            ECC_EC_D4 = ECC_EC_D4 + 1;
          end
        end
      end
    end

    /* Initialization */
    initial begin
      state = S_IDLE;
      dout = 0;
      En_lctr = 0;
      Gn_lctr = 0;
      Gn_hctr = 0;
      Wn_lctr = 0;
      PGn_lctr = 0;
      Wn_nectr = 0;

      cfgmode = 0;
      cfgaccess = 0;
      cfgdummy_addr = 0;
      cfgdummy_wr = 0;
      
      ECC_TE_D1 = DEF_ECC_TE;
      ECC_EI_D1 = DEF_ECC_EI;
      ECC_DI_D1 = DEF_ECC_DI;
      ECC_DO_D1 = DEF_ECC_DO;
      ECC_CR_D1 = DEF_ECC_CR;
      ECC_EC_D1 = DEF_ECC_EC;
      IFR_D1    = DEF_IFR;
      ODSR_D1   = DEF_ODSR;
      AWPR_D1   = DEF_AWPR;

      ECC_TE_D2 = DEF_ECC_TE;
      ECC_EI_D2 = DEF_ECC_EI;
      ECC_DI_D2 = DEF_ECC_DI;
      ECC_DO_D2 = DEF_ECC_DO;
      ECC_CR_D2 = DEF_ECC_CR;
      ECC_EC_D2 = DEF_ECC_EC;
      IFR_D2    = DEF_IFR;
      ODSR_D2   = DEF_ODSR;
      AWPR_D2   = DEF_AWPR;

      ECC_TE_D3 = DEF_ECC_TE;
      ECC_EI_D3 = DEF_ECC_EI;
      ECC_DI_D3 = DEF_ECC_DI;
      ECC_DO_D3 = DEF_ECC_DO;
      ECC_CR_D3 = DEF_ECC_CR;
      ECC_EC_D3 = DEF_ECC_EC;
      IFR_D3    = DEF_IFR;
      ODSR_D3   = DEF_ODSR;
      AWPR_D3   = DEF_AWPR;

      ECC_TE_D4 = DEF_ECC_TE;
      ECC_EI_D4 = DEF_ECC_EI;
      ECC_DI_D4 = DEF_ECC_DI;
      ECC_DO_D4 = DEF_ECC_DO;
      ECC_CR_D4 = DEF_ECC_CR;
      ECC_EC_D4 = DEF_ECC_EC;
      IFR_D4    = DEF_IFR;
      ODSR_D4   = DEF_ODSR;
      AWPR_D4   = DEF_AWPR;

    end

endmodule
